//=====================================================================
//
// Copyright Peter Schmitteckert, 2001
// http://www.schmitteckert.com
//
// all rights reserved
//
// pPublished under GPL v2 (GNPublic License version2)
//
// NOTE:
// There is absolutely no WARRANTY. This programm may create
// a wormhole transferring you to another galaxy.
// Use at your own risk.
//
//=====================================================================

#include "Ising.h"

#include <iostream>
#include <fstream>
#include <iomanip>
#include <stdlib.h>
#include <math.h>
#include <vector.h>

void  Ising2d::setParameterTemperature( const double TT )
{
    T =TT;
    calc_W();
};

Ising2d::Ising2d(int nx, int ny, double TT , double hh)
{
    int i;

    N_x      = (nx + BaseSize -1)/ BaseSize;
    if(N_x<2)
        N_x=2;

    N_xspins = N_x * BaseSize;

    N_y = (ny<3) ? 4 : ny;

    if( N_y & 1)
        ++N_y;

    N_yspins = N_y;
    N_spins = N_xspins*N_yspins;

    pChain = new IsingChain[N_y];
    pChain[0] = new BASE[N_y*N_x];

    if( (pChain==NULL) || (pChain[0] == NULL) )
    {
        cerr << "*** Out of Memory *** " << endl;
        exit(-1); // Should be improved
    }

    for(i=1; i<N_y;i++)
    {
        pChain[i] = pChain[i-1]+ N_x;
    }

    Chainb0 = new BASE[N_x];
    Chainb1 = new BASE[N_x];
    Chainb2 = new BASE[N_x];

    T = TT;
    H = h = hh;
    calc_W();
}

Ising2d::~Ising2d()
{ int i;

  delete [] Chainb0;
  delete [] Chainb1;
  delete [] Chainb2;
  delete [] (pChain[0]);
  delete [] pChain;
}

inline BASE Random(void)
{
    //static Zufall ZZZ(100003);
    return ( ( random()<<8) ^ random() );
}

void Ising2d::Randomize()
{
    for( int i=0; i<N_y; ++i)
        for(int j=0;j<N_x;++j) pChain[i][j] =  Random(); //0xaaaaaaaa;
}


#define MIN(a,b)  ( (a)<(b) ? (a) : (b) )

/*
// Metropolis Algorithmis
void Ising2d::calc_W()
{
	const double dc = double(0xffffffff);   // doesn't work on 64 Bit systems, ~0 ?

	// if(T>0) // T>0 wird vom Hauptprogramm granatiert
	{
		// Note: Ein Spinflip ndert den Betrag um 2/T
		const double w1 = 2./T;
		const double w2 = 2./T;

		// W[broken_bonds_x][[broken_bonds_y][ on site spin ]
		// cout << "Calculating Metropolis Weights using T="<<T<<" H="<< h << endl;

		W[0][0] = (BASE) (dc * MIN( (exp(( -w1*(1-0) -w2*( 1-0 ))) ),1.));
		W[0][1] = (BASE) (dc * MIN( (exp(( -w1*(1-0) -w2*( 1-1 ))) ),1.));
		W[0][2] = (BASE) (dc * MIN( (exp(( -w1*(1-0) -w2*( 1-2 ))) ),1.));
		W[1][0] = (BASE) (dc * MIN( (exp(( -w1*(1-1) -w2*( 1-0 ))) ),1.));
		W[1][1] = (BASE) (dc * MIN( (exp(( -w1*(1-1) -w2*( 1-1 ))) ),1.));
		W[1][2] = (BASE) (dc * MIN( (exp(( -w1*(1-1) -w2*( 1-2 ))) ),1.));
		W[2][0] = (BASE) (dc * MIN( (exp(( -w1*(1-2) -w2*( 1-0 ))) ),1.));
		W[2][1] = (BASE) (dc * MIN( (exp(( -w1*(1-2) -w2*( 1-1 ))) ),1.));
		W[2][2] = (BASE) (dc * MIN( (exp(( -w1*(1-2) -w2*( 1-2 ))) ),1.));

	cout << "W00: " << W[0][0] << " W11: " << W[1][1] << " W22:" << W[2][2] << "  Ref: " << exp( -w1 ) << endl;
	}
}
*/

// Glauber Dynamik
void Ising2d::calc_W()
{
//	const double dc = double(0xffffffff);   // doesn't work on 64 Bit systems, ~0 ?
	const double dc = double(RAND_MAX);   

	// if(T>0) // T>0 wird vom Hauptprogramm granatiert
	{
		// Note: Ein Spinflip ndert den Betrag um 2/T
		const double w1 = 1./T;
		const double w2 = 1./T;

		// W[broken_bonds_x][[broken_bonds_y][ on site spin ]
		// cout << "Calculating Metropolis Weights using T="<<T<<" H="<< h << endl;

		W[0][0] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-0) -w2*( 1-0 ))) ));
		W[0][1] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-0) -w2*( 1-1 ))) ));
		W[0][2] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-0) -w2*( 1-2 ))) ));
		W[1][0] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-1) -w2*( 1-0 ))) ));
		W[1][1] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-1) -w2*( 1-1 ))) ));
		W[1][2] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-1) -w2*( 1-2 ))) ));
		W[2][0] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-2) -w2*( 1-0 ))) ));
		W[2][1] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-2) -w2*( 1-1 ))) ));
		W[2][2] = (BASE) (dc * 0.5 * ( 1 + tanh(( -w1*(1-2) -w2*( 1-2 ))) ));

	cout << "W00: " << W[0][0] << " W11: " << W[1][1] << " W22:" << W[2][2] << "  Ref: " << exp( -w1 ) << "  " << RAND_MAX << endl;
	}
}
/*
//Heat-bath
void Ising2d::calc_W()
{ const double w1 = 2.;
  const double w2 = 2.;
  const double dc = double(0xffffffff);

  // W[broken_bonds_x][[broken_bonds_y][ on site spin ]
  W[0][0][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-0)+w2*( 1-0 )-h)/T) ));
  W[0][1][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-0)+w2*( 1-1 )-h)/T) ));
  W[0][2][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-0)+w2*( 1-2 )-h)/T) ));
  W[1][0][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-1)+w2*( 1-0 )-h)/T) ));
  W[1][1][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-1)+w2*( 1-1 )-h)/T) ));
  W[1][2][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-1)+w2*( 1-2 )-h)/T) ));
  W[2][0][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-2)+w2*( 1-0 )-h)/T) ));
  W[2][1][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-2)+w2*( 1-1 )-h)/T) ));
  W[2][2][0] = (BASE) (dc * 1. / (1. + exp(( w1*(1-2)+w2*( 1-2 )-h)/T) ));

  W[0][0][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-0)+w2*( 1-0 )+h)/T) ));
  W[0][1][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-0)+w2*( 1-1 )+h)/T) ));
  W[0][2][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-0)+w2*( 1-2 )+h)/T) ));
  W[1][0][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-1)+w2*( 1-0 )+h)/T) ));
  W[1][1][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-1)+w2*( 1-1 )+h)/T) ));
  W[1][2][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-1)+w2*( 1-2 )+h)/T) ));
  W[2][0][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-2)+w2*( 1-0 )+h)/T) ));
  W[2][1][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-2)+w2*( 1-1 )+h)/T) ));
  W[2][2][1] = (BASE) (dc * 1. / (1. + exp(( w1*(1-2)+w2*( 1-2 )+h)/T) ));
}
*/

// Sublattice A (10101010...)
inline void  Ising2d::update_A( IsingChain s, IsingChain by1, IsingChain by2 )
{
    int  x;
    BASE cl;       // left Base-element
    BASE c;        // old, new Spinconfiguration
    BASE cf;       // c-flip (1 -> flip)
    BASE bx;       // x-bonds, y-bonds
    BASE sx;       // verticale Summe
    BASE sy;       // horizontale Summe

    c = s[N_x-1];

    for ( x = 0; x<N_x; x++ )
    {
        cl = c; // left spins
        c = s[x]; // current spin block

        bx  = (c>>1);
        if (cl&1)
            bx |= Mask_Msb;
        bx ^= c;
        sx  = (bx & Mask_B) + ( (bx>>1) & Mask_B);

        sy  =  (((by1[x]) >>1) & Mask_B) + (((by2[x]) >>1) & Mask_B);

        //for( int cc=c>>1,cf=2; cf; cc>>=2, cf<<=2, sx>>=2, sy>>= 2)
        for( int cc=c>>1,cf=2; cf; cc>>=2, cf<<=2, sx>>=2, sy>>= 2)
        {
            // cout << " sx:"<<(sx&3)<<" sy:"<< (sy&3) <<"  --> W: "<<W[sx&3][sy&3]<<" ~~~ "<<double(W[sx&3][sy&3])/double(0xffffffff)<<endl;

            // Monte-Carlo Step
			BASE bbb;
			bbb = static_cast<BASE>(random());
            //cout << W[(sx&3)][(sy&3)] << '\t' << bbb << '\t' << W[(sx&3)][(sy&3)] - bbb<< endl;
			if ( W[(sx&3)][(sy&3)] >  bbb) // ::Random())
                c^= cf ;
        }
        s[x] = c;
    }
}

// Sublattice B (01010101...)
inline void  Ising2d::update_B( IsingChain s, IsingChain by1, IsingChain by2 )
{
    int  x;
    BASE cr;       // right Base-element
    BASE c;        // old, new Spinconfiguration
    BASE cf;       // c-flip (1 -> flip)
    BASE bx;       // x-bonds, y-bonds
    BASE sx;       // verticale Summe
    BASE sy;       // horizontale Summe

    c = s[0];

    for ( x = N_x-1; x>=0; --x )
    {
        cr = c;
        c = s[x];

        bx  = (c<<1);
        if (cr & Mask_Msb)
            bx |= 1;
        bx ^= c;
        sx  = (bx & Mask_B) + ( (bx>>1) & Mask_B);

        sy  =  ((by1[x]) & Mask_B) + ((by2[x]) & Mask_B);

        for( int cc=c,cf=1; cf; cf<<=2, cc>>=2,sx>>=2, sy>>= 2)
        {
            // Monte-Carlo Step
            if(W[(sx&3)][(sy&3)] >  static_cast<BASE>(random()) ) // ::Random())
                c^=cf;
        }
        s[x] = c;
    }
}


void Ising2d::Simulate( int N_Iterationen)
{
    int  y,j,l;
    int  m, mm=0;
    double h1=h;

    double temp;

    for(j=0;j< N_Iterationen; ++j)
    {

        // Sublattice A (10101010...)
        //              (01010101...)

        for(l=0; l<N_x;++l)
            Chainb0[l] = pChain[0][l] ^ pChain[N_y-1][l];

        for(l=0; l<N_x;++l)
            Chainb1[l] = pChain[0][l] ^ pChain[1][l];

        update_A( pChain[0], Chainb0, Chainb1 );

        for(y=1; y<N_y-1; ++y)
        {
            for(l=0; l<N_x;++l)
                Chainb2[l] = pChain[y][l] ^ pChain[y+1][l];
            update_B( pChain[y], Chainb1, Chainb2 );

            ++y;
            for(l=0; l<N_x;++l)
                Chainb1[l] = pChain[y][l] ^ pChain[y+1][l];
            update_A( pChain[y], Chainb2, Chainb1 );
        }
        update_B( pChain[N_y-1], Chainb0, Chainb1 );


        // Sublattice B (01010101...)
        //              (10101010...)

        for(l=0; l<N_x;++l)
            Chainb0[l] = pChain[0][l] ^ pChain[N_y-1][l];

        for(l=0; l<N_x;++l)
            Chainb1[l] = pChain[0][l] ^ pChain[1][l];

        update_B( pChain[0], Chainb0, Chainb1 );

        for(y=1; y<N_y-1; ++y)
        {
            for(l=0; l<N_x; ++l)
                Chainb2[l] = pChain[y][l] ^ pChain[y+1][l];
            update_A( pChain[y], Chainb1, Chainb2 );

            ++y;

            for(l=0; l<N_x;++l)
                Chainb1[l] = pChain[y][l] ^ pChain[y+1][l];

            update_B( pChain[y], Chainb2, Chainb1 );
        }
        update_A( pChain[N_y-1], Chainb0, Chainb1 );
    }
}


const BASE* Ising2d::GetDataPointer()
{
    return pChain[0];
}
